/* ****************************************************************************
 * Copyright: 2017-2025 RAYLASE GmbH
 * This source code is the proprietary confidential property of RAYLASE GmbH.
 * Reproduction, publication, or any form of distribution to
 * any party other than the licensee is strictly prohibited.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

#ifndef RLTB_TYPES_H_
#define RLTB_TYPES_H_

#include "rltbAPI.h"
#include "rltbTraceBufferConfig.h"

#include <stdint.h>

typedef enum
{
	rltbERROR = -1,
	rltbSUCCESS = 0
} rltbResult;

typedef int32_t rltbHandle;

#define rltbInvalidHandle rltbERROR

/// <summary>
/// TraceBuffer events
/// </summary>
/// <remarks>These events will only occur if the corresponding option is enabled using <see cref="RAYLASE.SPICE3.TraceBufferConfig"/> and applying the
/// configuration on the card using <see cref="TraceBufferClient.SetConfig(TraceBufferConfig)"/>.</remarks>
typedef enum
{
	/// <summary>
	/// This will occur on changes at <see cref="RAYLASE.SPICE3.IOPort.PortA"/>.
	/// The payload is 16 bits wide, one bit per PortA input.
	/// Payload bit 0 represents PortA0, bit 1 represents PortA1 ...
	/// </summary>
	PortA = 0x00,

	/// <summary>
	/// This will occur on changes at <see cref="RAYLASE.SPICE3.IOPort.PortB"/>.
	/// The payload is 16 bits wide, one bit per PortB input.
	/// Payload bit 0 represents PortB0, bit 1 represents PortB1 ...
	/// </summary>
	PortB = 0x01,

	/// <summary>
	/// This will occur on changes at <see cref="RAYLASE.SPICE3.IOPort.PortC"/>.
	/// The payload is 16 bits wide, one bit per PortC input.
	/// Payload bit 0 represents PortC0, bit 1 represents PortC1 ...
	/// </summary>
	PortC = 0x02,

	/// <summary>
	/// This will occur on changes at <see cref="RAYLASE.SPICE3.IOPort.PortD"/>.
	/// The payload is 16 bits wide, one bit per PortD input.
	/// Payload bit 0 represents PortD0, bit 1 represents PortD1 ...
	/// </summary>
	PortDLow = 0x03,

	/// <summary>
	/// This will occur on changes at <see cref="RAYLASE.SPICE3.IOPort.PortD"/>.
	/// The payload is 8 bits wide, one bit per PortD input.
	/// Payload bit 0 represents PortD16, bit 1 represents PortD17 ...
	/// </summary>
	PortDHigh = 0x04,

	/// <summary>
	/// This will occur on changes at <see cref="RAYLASE.SPICE3.IOPort.PortE"/>.
	/// The payload is 16 bits wide, one bit per PortE input.
	/// Payload bit 0 represents PortE0, bit 1 represents PortE1 ...
	/// </summary>
	PortELow = 0x05,

	/// <summary>
	/// This will occur on changes at <see cref="RAYLASE.SPICE3.IOPort.PortE"/>.
	/// The payload is 8 bits wide, one bit per PortE input.
	/// Payload bit 0 represents PortE16, bit 1 represents PortE17 ...
	/// </summary>
	PortEHigh = 0x06,

	/// <summary>
	/// This will occur on changes at <see cref="RAYLASE.SPICE3.IOPort.LaserIn"/>.
	/// The payload is 2 bits wide, one bit per LaserIn input.
	/// </summary>
	PortLaserIn = 0x07,

	/// <summary>
	/// This will occur on changes of bit 0 at <see cref="RAYLASE.SPICE3.IOPort.LaserIn"/>.
	/// The payload size is one bit.
	/// </summary>
	PortLaserIn0 = 0x08,

	/// <summary>
	/// This will occur on changes of bit 1 at <see cref="RAYLASE.SPICE3.IOPort.LaserIn"/>.
	/// The payload size is one bit.
	/// </summary>
	PortLaserIn1 = 0x09,

	/// <summary>
	/// This will occur on changes at <see cref="RAYLASE.SPICE3.IOPort.LaserOut"/>.
	/// The payload is 3 bits wide, one bit per LaserOut output.
	/// </summary>
	PortLaserOut = 0x0A,

	/// <summary>
	/// This will occur on changes of bit 0 at <see cref="RAYLASE.SPICE3.IOPort.LaserOut"/>.
	/// The payload size is one bit.
	/// </summary>
	PortLaserOut0 = 0x0B,

	/// <summary>
	/// This will occur on changes of bit 1 at <see cref="RAYLASE.SPICE3.IOPort.LaserOut"/>.
	/// The payload size is one bit.
	/// </summary>
	PortLaserOut1 = 0x0C,

	/// <summary>
	/// This will occur on changes of bit 2 at <see cref="RAYLASE.SPICE3.IOPort.LaserOut"/>.
	/// The payload size is one bit.
	/// </summary>
	PortLaserOut2 = 0x0D,

	/// <summary>
	/// This will occur on changes at <see cref="RAYLASE.SPICE3.PowerTarget.Dac0"/> output.
	/// The payload is 16 bits wide.
	/// A value of 0 represents 0V and a value of 65535 represents +10V at the DAC output.
	/// Be aware that the actual DAC resolution is only 12 bits, so the 4 LSB do not affect the voltage.
	/// </summary>
	Dac0 = 0x0E,

	/// <summary>
	/// This will occur on changes at <see cref="RAYLASE.SPICE3.PowerTarget.Dac1"/> output.
	/// The payload is 16 bits wide.
	/// A value of 0 represents 0V and a value of 65535 represents +10V at the DAC output.
	/// Be aware that the actual DAC resolution is only 12 bits, so the 4 LSB do not affect the voltage.
	/// </summary>
	Dac1 = 0x0F,

	/// <summary>
	/// This will occur when the laser output power changes.
	/// The payload contains all 16 bits.
	/// </summary>
	LaserPower16 = 0x10,

	/// <summary>
	/// This will occur when the laser output power changes its 8 MSB bits.
	/// The payload contains the 8 MSB bits.
	/// </summary>
	LaserPower8 = 0x11,

	/// <summary>
	/// This will occur when the laser output power changes its 4 MSB bits.
	/// The payload contains the 4 MSB bits.
	/// </summary>
	LaserPower4 = 0x12,

	/// <summary>
	/// This will occur when the laser output power changes its 2 MSB bits.
	/// The payload contains the 2 MSB bits.
	/// </summary>
	LaserPower2 = 0x13,

	/// <summary>
	/// This will occur when the laser output power changes its MSB bit.
	/// The payload contains the MSB bit.
	/// </summary>
	LaserPower1 = 0x14,

	/// <summary>
	/// This will occur on every change of <see cref="RAYLASE.SPICE3.LaserSignal.Fps"/>.
	/// The payload size is one bit.
	/// </summary>
	Fps = 0x15,

	/// <summary>
	/// This will occur on every change of <see cref="RAYLASE.SPICE3.LaserSignal.Gate"/>.
	/// The payload size is one bit.
	/// </summary>
	Gate = 0x16,

	/// <summary>
	/// This will occur on every change of <see cref="RAYLASE.SPICE3.LaserSignal.LM"/>.
	/// The payload size is one bit.
	/// </summary>
	Lm = 0x17,

	/// <summary>
	/// This will occur on every change of 'LaserTriggered'/>.
	/// The payload size is one bit.
	/// </summary>
	LaserTriggered = 0x18,

	/// <summary>
	/// This will occur on every change of <see cref="RAYLASE.SPICE3.Sfr0Bits.StartMark"/>.
	/// The payload size is one bit.
	/// </summary>
	StartMark = 0x19,

	/// <summary>
	/// This will occur on every change of <see cref="RAYLASE.SPICE3.Sfr0Bits.AbortMark"/>.
	/// The payload size is one bit.
	/// </summary>
	AbortMark = 0x1A,

	/// <summary>
	/// This will occur when a X-frame is sent to Head0.
	/// The data does not need to have changed for this event to occur so this event occurs every 10us.
	/// The payload size is 32 bits.
	/// The bits 19 to 0 contain the frame payload. If bit 28 is set the frame is a command frame otherwise a position frame.
	/// The position data is interpreted as a signed 20.
	/// </summary>
	Head0TxX = 0x20,

	/// <summary>
	/// This will occur when a Y-frame is sent to Head0.
	/// </summary>
	Head0TxY = 0x22,

	/// <summary>
	/// This will occur when a Z-frame is sent to Head0.
	/// </summary>
	Head0TxZ = 0x24,

	/// <summary>
	/// This will occur when a X-frame is received from Head0.
	/// The event payload size is 32 bits.
	/// When the XY2-100 interface is selected this event occurs every 10us regardless of a scan head is connected or not.
	/// When the SL2-100 interface is selected this event also occurs every 10us but will be omitted if no frame is received (e.g. no scan head is connected).
	/// For detailed payload description see "Bit Definitions" in <conceptualLink target="ee3edb0d-7e41-4e18-8b70-5d7a6916e9ba" />.
	/// </summary>
	Head0RxX = 0x28,

	/// <summary>
	/// This will occur when a Y-frame is received from Head0.
	/// </summary>
	Head0RxY = 0x2A,

	/// <summary>
	/// This will occur when a Z-frame is received from Head0.
	/// </summary>
	Head0RxZ = 0x2C,

	/// <summary>
	/// This will occur when a X-frame is sent to Head1.
	/// </summary>
	Head1TxX = 0x30,

	/// <summary>
	/// This will occur when a Y-frame is sent to Head1.
	/// </summary>
	Head1TxY = 0x32,

	/// <summary>
	/// This will occur when a Z-frame is sent to Head1.
	/// </summary>
	Head1TxZ = 0x34,

	/// <summary>
	/// This will occur when a X-frame is received from Head1.
	/// </summary>
	Head1RxX = 0x38,

	/// <summary>
	/// This will occur when a Y-frame is received from Head1.
	/// </summary>
	Head1RxY = 0x3A,

	/// <summary>
	/// This will occur when a Z-frame is received from Head1.
	/// </summary>
	Head1RxZ = 0x3C,

	/// <summary>
	/// This will occur when the laser output power changes.
	/// The payload contains all 16 bits.
	/// </summary>
	LaserPower0_16 = 0x40,

	/// <summary>
	/// This will occur when the laser output power changes its 8 MSB bits.
	/// The payload contains the 8 MSB bits.
	/// </summary>
	LaserPower0_8 = 0x41,

	/// <summary>
	/// This will occur when the laser output power changes its 4 MSB bits.
	/// The payload contains the 4 MSB bits.
	/// </summary>
	LaserPower0_4 = 0x42,

	/// <summary>
	/// This will occur when the laser output power changes its 2 MSB bits.
	/// The payload contains the 2 MSB bits.
	/// </summary>
	LaserPower0_2 = 0x43,

	/// <summary>
	/// This will occur when the laser output power changes its MSB bit.
	/// The payload contains the MSB bit.
	/// </summary>
	LaserPower0_1 = 0x44,

	/// <summary>
	/// This will occur when the laser output power changes.
	/// The payload contains all 16 bits.
	/// </summary>
	LaserPower1_16 = 0x45,

	/// <summary>
	/// This will occur when the laser output power changes its 8 MSB bits.
	/// The payload contains the 8 MSB bits.
	/// </summary>
	LaserPower1_8 = 0x46,

	/// <summary>
	/// This will occur when the laser output power changes its 4 MSB bits.
	/// The payload contains the 4 MSB bits.
	/// </summary>
	LaserPower1_4 = 0x47,

	/// <summary>
	/// This will occur when the laser output power changes its 2 MSB bits.
	/// The payload contains the 2 MSB bits.
	/// </summary>
	LaserPower1_2 = 0x48,

	/// <summary>
	/// This will occur when the laser output power changes its MSB bit.
	/// The payload contains the MSB bit.
	/// </summary>
	LaserPower1_1 = 0x49,

	/// <summary>
	/// This will occur when the SPI0 module received something.
	/// </summary>
	Spi0Rx = 0x60,

	/// <summary>
	/// This will occur when the SPI1 module received something.
	/// </summary>
	Spi1Rx = 0x61,

	/// <summary>
	/// This will occur when the SPI2 module received something.
	/// </summary>
	Spi2Rx = 0x62,

	/// <summary>
	/// This will occur when the SPI3 module received something.
	/// </summary>
	Spi3Rx = 0x63,

	/// <summary>
	/// This will occur on every change of the 'quadrature index' signal of module 0.
	/// </summary>
	Quadrature0I = 0x64,

	/// <summary>
	/// This will occur on every change of the 'quadrature A' signal of module 0.
	/// </summary>
	Quadrature0A = 0x65,

	/// <summary>
	/// This will occur on every change of the 'quadrature B' signal of module 0.
	/// </summary>
	Quadrature0B = 0x66,

	/// <summary>
	/// This will occur on every change of the 'quadrature index' signal of module 1.
	/// </summary>
	Quadrature1I = 0x67,

	/// <summary>
	/// This will occur on every change of the 'quadrature A' signal of module 1.
	/// </summary>
	Quadrature1A = 0x68,

	/// <summary>
	/// This will occur on every change of the 'quadrature B' signal of module 1.
	/// </summary>
	Quadrature1B = 0x69,

	/// <summary>
	/// This will occur on changes inside the <see cref="RAYLASE.SPICE3.TraceBufferConfig.IOEvents"/> configuration register.
	/// </summary>
	IOEvents = 0x78,

	/// <summary>
	/// This will occur on changes inside the <see cref="RAYLASE.SPICE3.TraceBufferConfig.PositionEvents"/> configuration register.
	/// </summary>
	PositionEvents = 0x79,

	/// <summary>
	/// This will occur on changes inside the <see cref="RAYLASE.SPICE3.TraceBufferConfig.ControlEvents"/> configuration register.
	/// </summary>
	ControlEvents = 0x7B,

	/// <summary>
	/// This will occur when a label that was added to the list gets executed. See <see cref="RAYLASE.SPICE3.TraceBufferConfig3.Label"/> for more details.
	/// Note that a label is 16 bits in size and left aligned in the 32 bit payload. Since one packed label event will always only contain one unpacking will
	/// not create two label events but shift the label to the first 16 bit of the payload.
	/// </summary>
	Label = 0x7E,

	/// <summary>
	/// This event type is only used internally and will never be returned.
	/// </summary>
	TimerOverflow = 0x7F,
} rltbTraceEventType;

/// <summary>
/// TraceBuffer event from the SP-ICE-3 card
/// </summary>
typedef struct
{
	/// <summary>
	/// TimeStamp at the time of the occurrence of this event
	/// </summary>
	/// <remarks>
	/// This contains a 56 bit timestamp. The value is the time interval since the last boot of the card in 15.625ns increments.
	/// </remarks>
	uint64_t Timestamp;

	/// <summary>
	/// Time in μs relative to the first event received.
	/// </summary>
	double Time;

	/// <summary>
	/// Type of the Event <see cref="rltbTraceEventType"/>.
	/// </summary>
	rltbTraceEventType EventType;

	/// <summary>
	/// Indicates if a timer overflow occurred. If this bit is set, events have been missed between this one and the last event of the same EventType.
	/// </summary>
	bool Overflow;

	/// <summary>
	/// The payload of the event.
	/// </summary>
	int32_t Value;
} rltbTraceEvent;

#endif // RLTB_TYPES_H_
